/*  Copyright 2017 Gibbon aka 'atsb'
    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
*/

#include "FLDungeon.hpp"
#include "../config/FLIncludeData.hpp"
#include "../templates/FLTMath.hpp"
#include "../terminal/FLTerminal.hpp"
#include "../core/FLFuncs.hpp"
#include "../core/FLTermCapIO.hpp"

static void build_proximity_ripple(void);
static void move_scared(int, int);
static void move_smart(int, int);
static void move_dumb(int, int);
static void mmove(int, int, int, int);
static int w1x[9], w1y[9];
static int tmp1, tmp2, tmp3, tmp4, distance_from;

/* list of monsters to move */
static struct foo {
	int x;
	int y;
	int smart;
} movelist[250];

/*
    fl_move_a_monster()     Routine to move the monsters toward the player

    This routine has the responsibility to determine which monsters are to
    move, and call movemt() to do the move.
    Returns no value.
*/
void
fl_move_a_monster(void) {
	int i, j, movecnt = 0, smart_count, min_int;
	if (cdesc[FL_HOLDMONST]) {
		return;  /* no action if monsters are held */
	}
	if (cdesc[FL_AGGRAVATE]) {	/* determine window of monsters to move */
		tmp1 = player_vertical_position - 5;
		tmp2 = player_vertical_position + 6;
		tmp3 = player_horizontal_position - 10;
		tmp4 = player_horizontal_position + 11;
		distance_from =
		    IDISTAGGR;	/* depth of intelligent monster movement */
	} else {
		tmp1 = player_vertical_position - 3;
		tmp2 = player_vertical_position + 4;
		tmp3 = player_horizontal_position - 5;
		tmp4 = player_horizontal_position + 6;
		distance_from =
		    IDISTNORM;	/* depth of intelligent monster movement */
	}
	if (level ==
	        0) {	/* if on outside level monsters can move in perimeter */
		if (tmp1 < 0) {
			tmp1 = 0;
		}
		if (tmp2 > FL_MAX_VERTICAL_POSITION) {
			tmp2 = FL_MAX_VERTICAL_POSITION;
		}
		if (tmp3 < 0) {
			tmp3 = 0;
		}
		if (tmp4 > FL_MAX_HORIZONTAL_POSITION) {
			tmp4 = FL_MAX_HORIZONTAL_POSITION;
		}
	} else {			/* if in a dungeon monsters can't be on the perimeter (wall there) */
		if (tmp1 < 1) {
			tmp1 = 1;
		}
		if (tmp2 > FL_MAX_VERTICAL_POSITION - 1) {
			tmp2 = FL_MAX_VERTICAL_POSITION - 1;
		}
		if (tmp3 < 1) {
			tmp3 = 1;
		}
		if (tmp4 > FL_MAX_HORIZONTAL_POSITION - 1) {
			tmp4 = FL_MAX_HORIZONTAL_POSITION - 1;
		}
	}
	smart_count = 0;
	min_int = 10; /* minimum monster intelligence to move smart */
	if (cdesc[FL_AGGRAVATE] || !cdesc[FL_STEALTH]) {
		for (j = tmp1; j < tmp2; j++)
			for (i = tmp3; i < tmp4; i++)
				if (monster_identification[i][j]) {
					movelist[movecnt].x = i;
					movelist[movecnt].y = j;
					if (monster[monster_identification[i][j]].intelligence > min_int) {
						movelist[movecnt].smart = 1;
						smart_count++;
					} else {
						movelist[movecnt].smart = 0;
					}
					movecnt++;
				}
	}
	/*  now move the monsters in the movelist.  If we have at least one
	    smart monster, build a proximity ripple and use it for all smart
	    monster movement.
	*/
	if (movecnt > 0) {
		if (cdesc[FL_SCAREMONST])
			for (i = 0; i < movecnt; i++) {
				move_scared(movelist[i].x, movelist[i].y);
			} else {
			if (smart_count > 0) {
				build_proximity_ripple();
				for (i = 0; i < movecnt; i++)
					if (movelist[i].smart) {
						move_smart(movelist[i].x, movelist[i].y);
					} else {
						move_dumb(movelist[i].x, movelist[i].y);
					}
			} else
				for (i = 0; i < movecnt; i++) {
					move_dumb(movelist[i].x, movelist[i].y);
				}
		}
	}
	/*  Also check for the last monster hit.  This is necessary to prevent
	    the player from getting free hits on a monster with long range
	    spells or when stealthed.
	*/
	if (cdesc[FL_AGGRAVATE] || !cdesc[FL_STEALTH]) {
		/*  If the last monster hit is within the move window, its already
		    been moved.
		*/
		if (((lasthx < tmp3 || lasthx >= tmp4) ||
		        (lasthy < tmp1 || lasthy >= tmp2))
		        && monster_identification[lasthx][lasthy]) {
			if (cdesc[FL_SCAREMONST]) {
				move_scared(lasthx, lasthy);
			} else if (monster[monster_identification[lasthx][lasthy]].intelligence >
			           min_int) {
				if (smart_count == 0) {
					build_proximity_ripple();
				}
				move_smart(lasthx, lasthy);
			} else {
				move_dumb(lasthx, lasthy);
			}
			lasthx = w1x[0];	/* make sure the monster gets moved again */
			lasthy = w1y[0];
		}
	} else {
		/*  If the last monster hit is within the move window, and not
		    asleep due to stealth, then it has already been moved.
		    Otherwise (monster outside window, asleep due to stealth),
		    move the monster and update the lasthit x,y position.
		*/
		if ((((lasthx < tmp3 || lasthx >= tmp4) ||
		        (lasthy < tmp1 || lasthy >= tmp2)) &&
		        monster_identification[lasthx][lasthy]) || !stealth[lasthx][lasthy]) {
			if (cdesc[FL_SCAREMONST]) {
				move_scared(lasthx, lasthy);
			} else if (monster[monster_identification[lasthx][lasthy]].intelligence >
			           min_int) {
				if (smart_count == 0) {
					build_proximity_ripple();
				}
				move_smart(lasthx, lasthy);
			} else {
				move_dumb(lasthx, lasthy);
			}
			lasthx = w1x[0];	/* make sure the monster gets moved again */
			lasthy = w1y[0];
		}
	}
}

int screen[FL_MAX_HORIZONTAL_POSITION][FL_MAX_VERTICAL_POSITION];		/* proximity ripple storage */

/*  queue for breadth-first 'search' build of proximity ripple.
*/
static struct queue_entry {
	int x;
	int y;
	int distance_from;
} queue[MAX_QUEUE];
static int queue_head = 0;
static int queue_tail = 0;

/*  put a location on the proximity ripple queue
*/
void fl_put_queue(int x, int y, int d) {
	queue[queue_tail].x = (x) ;
	queue[queue_tail].y = (y) ;
	queue[queue_tail].distance_from = (d);
	queue_tail++;
	if (queue_tail == MAX_QUEUE)
		queue_tail = 0;
}

/*  take a location from the proximity ripple queue
*/
void fl_get_queue(int x, int y, int d) {
	(x) = queue[queue_head].x;
	(y) = queue[queue_head].y;
	(d) = queue[queue_head].distance_from;
	queue_head++;
	if (queue_head == MAX_QUEUE)
		queue_head = 0;
}

/*
    For smart monster movement, build a proximity ripple from the player's
    position, out to a 'distance' of 20.  For example:

    W 5 4 4 W W X    Player is at position marked 1
    W 5 W 3 3 W W    W is a wall.  Monsters will attempt
    W 6 W 2 W 4 W    to move to a location with a smaller
    W 7 W 1 W 5 W    value than their current position.
    W 8 W W W 6 W    Note that a monster at location X
    W 9 9 8 7 7 7    will not move at all.
    W W W 8 W W W
*/
static void
build_proximity_ripple(void) {
	int xl, yl, xh, yh;
	int k, m, z, tmpx, tmpy;
	int curx, cury, curdist;
	xl = tmp3 - 2;
	yl = tmp1 - 2;
	xh = tmp4 + 2;
	yh = tmp2 + 2;
	fl_verify_bound_coordinates(&xl, &yl);
	fl_verify_bound_coordinates(&xh, &yh);
	for (k = yl; k <= yh; k++)
		for (m = xl; m <= xh; m++) {
			switch (object_identification[m][k]) {
				case OWALL:
				case OPIT:
				case OTRAPARROW:
				case ODARTRAP:
				case OCLOSEDDOOR:
				case OTRAPDOOR:
				case OTELEPORTER:
					screen[m][k] = 127;
					break;
				case OENTRANCE:
					if (level == 1) {
						screen[m][k] = 127;
					} else {
						screen[m][k] = 0;
					}
					break;
				default:
					screen[m][k] = 0;
					break;
			};
		}
	screen[player_horizontal_position][player_vertical_position] = 1;
	/* now perform proximity ripple from player_horizontal_position,player_vertical_position to monster */
	xl = tmp3 - 1;
	yl = tmp1 - 1;
	xh = tmp4 + 1;
	yh = tmp2 + 1;
	fl_verify_bound_coordinates(&xl, &yl);
	fl_verify_bound_coordinates(&xh, &yh);
	curx = 0;
	cury = 0;
	curdist = 0;
	fl_put_queue(player_horizontal_position, player_vertical_position, 1);
	do {
		fl_get_queue(curx, cury, curdist);
		/*  test all spots around the current one being looked at.
		*/
		if ((curx >= xl && curx <= xh) && (cury >= yl
		                                   && cury <= yh)) {
			for (z = 1; z < 9; z++) {
				tmpx = curx + diroffx[z];
				tmpy = cury + diroffy[z];
				fl_verify_bound_coordinates(&tmpx, &tmpy);
				if (screen[tmpx][tmpy] == 0) {
					screen[tmpx][tmpy] = curdist + 1;
					fl_put_queue(tmpx, tmpy, curdist + 1);
				}
			}
		}
	}
	/* check for the proximity ripple queue being empty */
	while (queue_head != queue_tail);
}

/*
    Move scared monsters randomly away from the player position.
*/
static void
move_scared(int i, int j) {
	int xl, yl, tmp;
	/*  check for a half-speed monster, and check if not to move.  Could be
	    done in the monster list build.
	*/
	switch (monster_identification[i][j]) {
		case TROGLODYTE:
		case HOBGOBLIN:
		case METAMORPH:
		case XVART:
		case FL_INVISIBLESTALKER:
		case ICELIZARD:
			if ((gtime & 1) == 1) {
				return;
			}
	};
	if ((xl = i + TRnd(3) - 2) < 0) {
		xl = 0;
	}
	if (xl >= FL_MAX_HORIZONTAL_POSITION) {
		xl = FL_MAX_HORIZONTAL_POSITION - 1;
	}
	if ((yl = j + TRnd(3) - 2) < 0) {
		yl = 0;
	}
	if (yl >= FL_MAX_VERTICAL_POSITION) {
		yl = FL_MAX_VERTICAL_POSITION - 1;
	}
	if ((tmp = object_identification[xl][yl]) != OWALL)
		if (monster_identification[xl][yl] == 0)
			if ((monster_identification[i][j] != VAMPIRE) || (tmp != OMIRROR))
				if (tmp != OCLOSEDDOOR) {
					mmove(i, j, xl, yl);
				}
}



/*
    Move monsters that are moving intelligently, using the proximity
    ripple.  Attempt to move to a position in the proximity ripple
    that is closer to the player.

    Parameters: the X,Y position of the monster to be moved.
*/
static void
move_smart(int i, int j) {
	int x, y, z;
	/*  check for a half-speed monster, and check if not to move.  Could be
	    done in the monster list build.
	*/
	switch (monster_identification[i][j]) {
		case TROGLODYTE:
		case HOBGOBLIN:
		case METAMORPH:
		case XVART:
		case FL_INVISIBLESTALKER:
		case ICELIZARD:
			if ((gtime & 1) == 1) {
				return;
			}
	};
	/*  find an adjoining location in the proximity ripple that is
	    closer to the player (has a lower value) than the monster's
	    current position.
	*/
	if (monster_identification[i][j] != VAMPIRE)
		for (z = 1; z < 9; z++) {	/* go around in a circle */
			x = i + diroffx[z];
			y = j + diroffy[z];
			if (screen[x][y] < screen[i][j])
				if (!monster_identification[x][y]) {
					mmove(i, j, w1x[0] = x, w1y[0] = y);
					return;
				}
		} else
		/*  prevent vampires from moving onto mirrors
		*/
		for (z = 1; z < 9; z++) {	/* go around in a circle */
			x = i + diroffx[z];
			y = j + diroffy[z];
			if ((screen[x][y] < screen[i][j])
			        && (object_identification[x][y] != OMIRROR))
				if (!monster_identification[x][y]) {
					mmove(i, j, w1x[0] = x, w1y[0] = y);
					return;
				}
		}
}




/*
    For monsters that are not moving in an intelligent fashion.  Move
    in a direct fashion toward the player's current position.

    Parameters: the X,Y position of the monster to move.
*/
static void
move_dumb(int i, int j) {
	int xl, yl, xh, yh;
	int k, m, tmp, tmpd, tmpx, tmpy;
	/*  check for a half-speed monster, and check if not to move.  Could be
	    done in the monster list build.
	*/
	switch (monster_identification[i][j]) {
		case TROGLODYTE:
		case HOBGOBLIN:
		case METAMORPH:
		case XVART:
		case FL_INVISIBLESTALKER:
		case ICELIZARD:
			if ((gtime & 1) == 1) {
				return;
			}
	};
	/* dumb monsters move here */
	/*  set up range of spots to check.  instead of checking all points
	    around the monster, only check those closest to the player.  For
	    example, if the player is up and right of the monster, check only
	    the three spots up and right of the monster.
	*/
	xl = i - 1;
	yl = j - 1;
	xh = i + 2;
	yh = j + 2;
	if (i < player_horizontal_position) {
		xl++;
	} else if (i > player_horizontal_position) {
		--xh;
	}
	if (j < player_vertical_position) {
		yl++;
	} else if (j > player_vertical_position) {
		--yh;
	}
	if (xl < 0) {
		xl = 0;
	}
	if (yl < 0) {
		yl = 0;
	}
	if (xh > FL_MAX_HORIZONTAL_POSITION) {
		xh = FL_MAX_HORIZONTAL_POSITION;  /* FL_MAX_HORIZONTAL_POSITION OK; loop check below is <, not <= */
	}
	if (yh > FL_MAX_VERTICAL_POSITION) {
		yh = FL_MAX_VERTICAL_POSITION;  /* FL_MAX_VERTICAL_POSITION OK; loop check below is <, not <= */
	}
	/*  check all spots in the range.  find the one that is closest to
	    the player.  if the monster is already next to the player, exit
	    the check immediately.
	*/
	tmpd = 10000;
	tmpx = i;
	tmpy = j;
	for (k = xl; k < xh; k++)
		for (m = yl; m < yh; m++)
			if (k == player_horizontal_position && m == player_vertical_position) {
				tmpd = 1;
				tmpx = k;
				tmpy = m;
				break;		/* exitloop */
			} else if ((object_identification[k][m] != OWALL) &&
			           (object_identification[k][m] != OCLOSEDDOOR) &&
			           ((monster_identification[k][m] == 0) || ((k == i) && (m == j))) &&
			           ((monster_identification[i][j] != VAMPIRE) || (object_identification[k][m] != OMIRROR))) {
				tmp = (player_horizontal_position - k) * (player_horizontal_position - k) + (player_vertical_position - m) *
				      (player_vertical_position - m);
				if (tmp < tmpd) {
					tmpd = tmp;
					tmpx = k;
					tmpy = m;
				}			/* end if */
			}			/* end if */
	/*  we have finished checking the spaces around the monster.  if
	    any can be moved on and are closer to the player than the
	    current location, move the monster.
	*/
	if ((tmpd < 10000) && ((tmpx != i) || (tmpy != j))) {
		mmove(i, j, tmpx, tmpy);
		w1x[0] = tmpx;		/* for last monster hit */
		w1y[0] = tmpy;
	} else {
		w1x[0] = i;		/* for last monster hit */
		w1y[0] = j;
	}
}				/* end move_dumb() */




/*
    mmove(x,y,xd,yd)    Function to actually perform the monster movement
       int x,y,xd,yd;

    Enter with the from coordinates in (x,y) and the destination coordinates
    in (xd,yd).
*/
static void
mmove(int aa, int bb, int cc, int dd) {
	int tmp, i, flag;
	const char *who = "";
	const char *p = "";
	flag = 0;			/* set to 1 if monster hit by arrow trap */
	if ((cc == player_horizontal_position) && (dd == player_vertical_position)) {
		fl_hit_the_player(aa, bb);
		return;
	}
	i = object_identification[cc][dd];
	if ((i == OPIT) || (i == OTRAPDOOR))
		switch (monster_identification[aa][bb]) {
			case BAT:
			case EYE:
			case SPIRITNAGA:
			case PLATINUMDRAGON:
			case WRAITH:
			case VAMPIRE:
			case SILVERDRAGON:
			case POLTERGEIST:
			case DEMONLORD:
			case DEMONLORD + 1:
			case DEMONLORD + 2:
			case DEMONLORD + 3:
			case DEMONLORD + 4:
			case DEMONLORD + 5:
			case DEMONLORD + 6:
			case DEMONPRINCE:
				break;
			default:
				monster_identification[aa][bb] = 0;	/* fell in a pit or trapdoor */
		};
	tmp = monster_identification[aa][bb];
	monster_identification[cc][dd] = tmp;
	if (i == FL_OBJECT_SPHERE_OF_ANNIHILATION) {
		if (tmp >= DEMONLORD + 3) {	/* demons dispel spheres */
			fl_termcap_cursor_position(1, 24);
			lprintf("\nThe %s dispels the sphere!", monster[tmp].name);
			fl_remove_sphere_of_annihilation(cc, dd);	/* delete the sphere */
		} else {
			monster_identification[cc][dd] = i = tmp = 0;
		}
	}
	stealth[cc][dd] = 1;
	if ((monster_hit_points[cc][dd] = monster_hit_points[aa][bb]) < 0) {
		monster_hit_points[cc][dd] = 1;
	}
	monster_identification[aa][bb] = 0;
	if (tmp == LEPRECHAUN)
		switch (i) {
			case OGOLDPILE:
			case OMAXGOLD:
			case OKGOLD:
			case ODGOLD:
			case ODIAMOND:
			case ORUBY:
			case OEMERALD:
			case OSAPPHIRE:
				object_identification[cc][dd] = 0;	/* leprechaun takes gold */
		};
	if (tmp == TROLL)		/* if a troll regenerate him */
		if ((gtime & 1) == 0)
			if (monster[tmp].hitpoints > monster_hit_points[cc][dd]) {
				monster_hit_points[cc][dd]++;
			}
	if (i == OTRAPARROW) {	/* arrow hits monster */
		who = "An arrow";
		if ((monster_hit_points[cc][dd] -= TRnd(10) + level) <= 0) {
			monster_identification[cc][dd] = 0;
			flag = 2;
		} else {
			flag = 1;
		}
	}
	if (i == ODARTRAP) {	/* dart hits monster */
		who = "A dart";
		if ((monster_hit_points[cc][dd] -= TRnd(6)) <= 0) {
			monster_identification[cc][dd] = 0;
			flag = 2;
		} else {
			flag = 1;
		}
	}
	if (i == OTELEPORTER) {	/* monster hits teleport trap */
		flag = 3;
		fl_fill_dungeon_with_monsters(monster_identification[cc][dd]);
		monster_identification[cc][dd] = 0;
	}
	if (cdesc[FL_BLINDCOUNT]) {
		return;  /* if blind don't show where monsters are   */
	}
	if (been_here_before[cc][dd] & HAVESEEN) {
		p = 0;
		if (flag) {
			fl_termcap_cursor_position(1, 24);
		}
		switch (flag) {
			case 1:
				p = "\n%s hits the %s";
				break;
			case 2:
				p = "\n%s hits and kills the %s";
				break;
			case 3:
				p = "\nThe %s%s gets teleported";
				who = "";
				break;
		};
		if (p) {
			lprintf(p, who, monster[tmp].name);
		}
	}
	/*  if (y_larn_rep>1) { been_here_before[aa][bb] &= 2;  been_here_before[cc][dd] &= 2; return; } */
	if (been_here_before[aa][bb] & HAVESEEN) {
		fl_show_designated_cell_only(aa, bb);
	}
	if (been_here_before[cc][dd] & HAVESEEN) {
		fl_show_designated_cell_only(cc, dd);
	}
}
